// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "ini.h"
#include "resource.h"
#include "stringCommons.h"
#include <fstream>

class IniLine {
public:
	size_t number;
	string name;
	string value;

	IniLine(size_t aNumber, const string& aName, const string& aValue)
		: number(aNumber), name(aName), value(aValue) {};
};

//This class is intended to map a global variable to the .ini file
class IniVariable {
public:
	IniVariable(const string& aName) : name(aName) {}
	virtual bool ReadFromString(const string& in) = 0;  //Returns true on success
	friend ostream& operator<<(ostream& str, const IniVariable& aVar) {
		return aVar.Print(str); }
	const string& GetName() const { return name; }
protected:
	virtual ostream& Print(ostream& str) const = 0;
private:
	string name;
};

#define INI_VARIABLE_CLASS(name, type) \
class Ini##name : public IniVariable {\
public:\
	Ini##name(const string& aName, type& aRef) : IniVariable(aName), ref(aRef) {}\
	bool ReadFromString(const string& in);\
protected:\
	ostream& Print(ostream& str) const;\
private:\
	type& ref;\
	Ini##name& operator=(const Ini##name&);\
};
#define INI_VARIABLE_FUNC(name, type) \
	IniVariable *newIniVariable(const string& aName, type& aRef) {\
	return new Ini##name(aName, aRef); }
#define INI_VARIABLE_BOTH(name, type) \
	INI_VARIABLE_CLASS(name, type) INI_VARIABLE_FUNC(name, type)

INI_VARIABLE_BOTH(Bool, bool);
bool IniBool::ReadFromString(const string& in) {
	if(in == "true" || in == "1")
		ref = true;
	else if(in == "false" || in == "0")
		ref = false;
	else
		return false;
	return true;
}
ostream& IniBool::Print(ostream& str) const {
	return str << ref;
}

INI_VARIABLE_BOTH(UInt, UINT);
bool IniUInt::ReadFromString(const string& in) {
	istringstream str(in);
	str >> ref;
	return str.good();
}
ostream& IniUInt::Print(ostream& str) const {
	return str << ref;
}

INI_VARIABLE_BOTH(String, string);
bool IniString::ReadFromString(const string& in) {
	ref = in;
	return true;
}
ostream& IniString::Print(ostream& str) const {
	return str << ref;
}

INI_VARIABLE_BOTH(Byte, BYTE);
bool IniByte::ReadFromString(const string& in) {
	istringstream str(in);
	UINT uint;
	str >> uint;
	if(str.fail() || uint > 255)
		return false;
	ref = (BYTE)uint;
	return true;
}
ostream& IniByte::Print(ostream& str) const {
	return str << UINT(ref);
}

INI_VARIABLE_CLASS(IP, UINT);
bool IniIP::ReadFromString(const string& in) {
	unsigned int a, b, c, d;
	if(sscanf_s(in.c_str(), "%u.%u.%u.%u", &a, &b, &c, &d) != 4)
		return false;
	if(a > 255 || b > 255 || c > 255 || d > 255)
		return false;
	MYASSERT(sizeof(in_addr) == sizeof(ref));
	in_addr& ia = MAKE(in_addr, ref);
	ia.S_un.S_un_b.s_b1 = (BYTE)a;
	ia.S_un.S_un_b.s_b2 = (BYTE)b;
	ia.S_un.S_un_b.s_b3 = (BYTE)c;
	ia.S_un.S_un_b.s_b4 = (BYTE)d;
	return true;
}
ostream& IniIP::Print(ostream& str) const {
	in_addr& ia = MAKE(in_addr, ref);
	return str << ia.S_un.S_un_b.s_b1 << "." << ia.S_un.S_un_b.s_b2 << "." <<
		ia.S_un.S_un_b.s_b3 << "." << ia.S_un.S_un_b.s_b4;
}

class IniEnum : public IniVariable {
public:
	IniEnum(string aName, int& aRef, const char** aStrings, int aNStrings)
		: IniVariable(aName), ref(aRef), strings(aStrings), nstrings(aNStrings) {}
	bool ReadFromString(const string& in);
	ostream& Print(ostream& str) const;
private:
	int& ref;
	const char** strings;
	const int nstrings;
	IniEnum& operator=(const IniEnum&);
};
#define NEW_INI_ENUM(name, var, strings)\
	(new IniEnum(name, *(int*)&(var), strings, sizeof(strings) / sizeof(void*)))

bool IniEnum::ReadFromString(const string& in) {
	for(int i=0; i<nstrings; i++) {
		if(in == strings[i]) {
			ref = i;
			return true;
		}
	}
	return false;
}
ostream& IniEnum::Print(ostream& str) const {
	MYASSERT(ref < nstrings);
	return str << strings[ref];
}

IniSystem::IniSystem(const string& aFilename) : mFilename(aFilename) {
#define STANDARD_VARS(macro) \
	macro(degub_run)\
	macro(beep)\
	macro(recompiler)\
	macro(rec_disasm)\
	macro(lowmem_log)\
	macro(cache_enabled)\
	macro(advanced_mode)\
	macro(log_interrupts)\
	macro(dumpmem)\
	macro(bouehr)\
	macro(dump_audio)\
	macro(quiet)\
	macro(degub_skip)\
	macro(autopause)\
	macro(log_audio)\
	macro(iloop)\
	macro(audio_buffer_min_size_ms)\
	macro(software_yuyv)\
	macro(gp_dtex)\
	macro(gp_dtexraw)\
	macro(gp_wireframe)\
	macro(gp_flat)\
	macro(gp_white)\
	macro(gp_log)\
	macro(edo_osr)\
	macro(use_mdvd)\
	macro(mdvd_filename)\
	macro(frame_limit)\
	macro(rec_log)\
	macro(exi_log)\
	macro(memcard_log)\
	macro(bba_log)\
	macro(si_log)\
	macro(always_on_top)\
	macro(dsp_log)\
	macro(dsp_hle)\
	macro(gp_pixelshader)\
	macro(rename_degub)\
	macro(degub_on)\
	macro(nerf_sc)\
	macro(dvd_log)\
	macro(gp_dlrec)\
	macro(separate_emudebug_file)\

#define PAD_VARS(macro) macro(reset_button) macro(pad_half) macro(pad_quarter)

	//#define IP_VARS(macro) \
	//macro(ip_src)\
	//macro(ip_dst_old)\
	//macro(ip_dst_new)\
	//macro(mac_address)\

#define STANDARD_VAR(name) mVars.push_back(newIniVariable(#name, g::name));

	STANDARD_VARS(STANDARD_VAR);

	mVars.push_back(newIniVariable("high_refresh_rate", g::hrr));
	mVars.push_back(NEW_INI_ENUM("timing_mode", g::timing_mode, g::str_tm));

	mVars.push_back(NEW_INI_ENUM("memcard1.content",
		g::memcardslot_info[0].content, str_MSC_short));
	mVars.push_back(newIniVariable("memcard1.caching",
		g::memcardslot_info[0].caching));
	mVars.push_back(newIniVariable("memcard1.filename",
		g::memcardslot_info[0].filename));
	mVars.push_back(NEW_INI_ENUM("memcard2.content",
		g::memcardslot_info[1].content, str_MSC_short));
	mVars.push_back(newIniVariable("memcard2.caching",
		g::memcardslot_info[1].caching));
	mVars.push_back(newIniVariable("memcard2.filename",
		g::memcardslot_info[1].filename));

	PAD_VARS(STANDARD_VAR);

#define INI_PAD_VKEYS(macro) \
	macro(a) macro(b) macro(x) macro(y) macro(z) macro(start) macro(l) macro(r)\
	macro(aup) macro(aleft) macro(adown) macro(aright)\
	macro(dup) macro(dleft) macro(ddown) macro(dright)\
	macro(cup) macro(cleft) macro(cdown) macro(cright)

#define INI_PAD_VKEY(id) str.str(""); str << "pad" << i << #id;\
	mVars.push_back(newIniVariable(str.str(), g::pad_sc[i].id));

	for(int i=0; i<PAD_NPADS; i++) {
		ostringstream str;
		str << "pad" << i << "_on";
		mVars.push_back(newIniVariable(str.str(), g::pad_on[i]));

		INI_PAD_VKEYS(INI_PAD_VKEY);
	}
}

bool IniSystem::WriteIniFile() {
	if(mVars.size() == 0)
		FAIL(UE_INI_NOT_INITIALIZED);

	ofstream file(mFilename.c_str());
	if(!file.good())
		FAIL(UE_INI_CANNOT_OPEN_FILE);

	for(size_t i=0; i<mVars.size(); i++) {
		const IniVariable& var = *mVars[i];
		file << var.GetName() << " = " << var << "\n";
	}

	return true;
}

IniSystem::~IniSystem() {
	for(size_t i=0; i<mVars.size(); i++) {
		delete mVars[i];
	}
}

bool IniSystem::DoIniFile() {
	vector<const IniLine> lines;
	bool rewrite=false;

	//Read all lines in .ini
	ifstream file(mFilename.c_str());
	size_t line = 1;
	while(file.good()) {
		string buffer, name, value;
		if(!getline(file, buffer).good())
			break;

		istringstream istr(buffer);
		istr >> name >> "=" >> value;
		MYASSERT(istr.eof());
		if(!istr.bad()) {
			lines.push_back(IniLine(line, name, value));
		} else {
			ostringstream ostr;
			ostr << "Error on line " << line << " in INI file: " << buffer;
			BFE(ostr.str());
			rewrite = true;
		}
		line++;
	}
	file.close();

	//Search for variables, read the ones that're there,
	//set defaults for the ones that aren't.
	line = 0;
	for(size_t i=0; i<mVars.size(); i++) {
		for(size_t d=0; d<lines.size(); d++) {
			if(lines[d].name == mVars[i]->GetName()) {
				if(!mVars[i]->ReadFromString(lines[d].value))
					rewrite = true;
				break;
			}
		}
		line++;
	}

	//Rewrite file if neccesary
	if(rewrite || lines.size() != line) {
		TGLE(WriteIniFile());
	}

	return true;
}
